-- ########################################################################
--
-- FunctionalForms, a combinator library for building wxHaskell forms
--
-- For a tutorial and more information, visit
-- http://www.cs.ru.nl/~sandr/FunctionalForms
--
-- The functions you'll need for defining forms are mainly:
-- (*) atomic forms; defined in this module
-- (*) form layout combinators; defined in the module FFormsLayout

module FForms
	( module Graphics.UI.WX
	, module Refs
	, module FFormsLayout
	, FForm
	, run_in_dialog
	, textEntry'
	, entry'
	, textCtrl'
	, textCtrlRich'
	, radioBox'
	, choice'
	, checkBox'
	, spinCtrl'
	, singleListBox'
	, multiListBox'
	, hslider'
	, vslider'
	, comboBox'
	, listCtrl'
	, moreButton'
	, tab2
	, tab3
	, (*-)
	, (*|)
	)
	where

import Refs
import FFormsMonad
import FFormsLayout
import Graphics.UI.WX
import Graphics.UI.WXCore



-- ##########  atomic forms  ###############

-- auxiliary function for concisely defining atomic forms
return_tuple app attr control =
	return
		( widget control
		, get control attr >>= return . app . const
		)

textEntry' :: [Prop (TextCtrl ())] -> Ref cx String -> FForm cx w Layout
textEntry' props (Ref val app) = FForm $ \w cx ->
	textEntry w ((text := val cx):props)
	>>= return_tuple app text

entry' = textEntry'

textCtrl' :: [Prop (TextCtrl ())] -> Ref cx String -> FForm cx w Layout
textCtrl' props (Ref val app) = FForm $ \w cx ->
	textCtrl w ((text := val cx):props)
	>>= return_tuple app text

textCtrlRich' :: [Prop (TextCtrl ())] -> Ref cx String -> FForm cx w Layout
textCtrlRich' props (Ref val app) = FForm $ \w cx ->
	textCtrlRich w ((text := val cx):props)
	>>= return_tuple app text

radioBox' :: Orientation -> [String] -> [Prop (RadioBox ())] -> Ref cx Int -> FForm cx w Layout
radioBox' ori labels props (Ref val app) = FForm $ \w cx ->
	radioBox w ori labels ((selection := val cx):props)
	>>= return_tuple app selection

choice' :: [Prop (Choice ())] -> Ref cx Int -> FForm cx w Layout
choice' props (Ref val app) = FForm $ \w cx ->
	choice w (props ++ [selection := val cx])
	>>= return_tuple app selection

checkBox' :: [Prop (CheckBox ())] -> Ref cx Bool -> FForm cx w Layout
checkBox' props (Ref val app) = FForm $ \w cx ->
	checkBox w ((checked := val cx):props)
	>>= return_tuple app checked


spinCtrl' :: Int -> Int -> [Prop (SpinCtrl ())] -> Ref cx Int -> FForm cx w Layout
spinCtrl' min max props (Ref val app) = FForm $ \w cx ->
	spinCtrl w min max ((selection := val cx):props)
	>>= return_tuple app selection
	

singleListBox' :: [Prop (SingleListBox ())] -> Ref cx Int -> FForm cx w Layout
singleListBox' props (Ref val app) = FForm $ \w cx ->
	singleListBox w (props ++ [selection := val cx])
	>>= return_tuple app selection

multiListBox' :: [Prop (MultiListBox ())] -> Ref cx [Int] -> FForm cx w Layout
multiListBox' props (Ref val app) = FForm $ \w cx ->
	multiListBox w (props ++ [selections := val cx])
	>>= return_tuple app selections

hslider' :: Bool -> Int -> Int -> [Prop (Slider ())] -> Ref cx Int -> FForm cx w Layout
hslider' showlabels min max props (Ref val app) = FForm $ \w cx ->
	hslider w showlabels min max ((selection := val cx):props)
	>>= return_tuple app selection

vslider' :: Bool -> Int -> Int -> [Prop (Slider ())] -> Ref cx Int -> FForm cx w Layout
vslider' showlabels min max props (Ref val app) = FForm $ \w cx ->
	vslider w showlabels min max ((selection := val cx):props)
	>>= return_tuple app selection

comboBox' :: [Prop (ComboBox ())] -> Ref cx String -> FForm cx w Layout
comboBox' props (Ref val app) = FForm $ \w cx ->
	do
		c <- comboBox w ((text := val cx):props)
		return_tuple app text c

listCtrl' :: [Prop (ListCtrl ())] -> Ref cx [[String]] -> FForm cx w Layout
listCtrl' props (Ref val app) = FForm $ \w cx ->
	listCtrl w (props ++ [items := val cx])
	>>= return_tuple app items



-- ##### Ref->FForm combinators ########

{-
moreButton' places a button on the form dialog which opens another form dialog
for editing 'advanced' options: parts of the subject type which are normally hidden
from editing.
-}
moreButton' ::
	(Ref t t -> FForm t (CPanel ()) Layout) ->
	Ref cx t ->
	FForm cx w Layout
moreButton' to_form (Ref v a) = FForm $ \w cx ->
	do	var <- varCreate (v cx)
		let rundialog =
			do	s <- varGet var
				s' <- run_in_dialog w to_form s
				varSet var s'
		b <- button w [text := "More...", on command := rundialog]
		return (widget b, do n <- varGet var; return $ a (const n))

-- *- places two ref->forms next to each other and joins their subject types
-- into a pair
infixr 5 *-
form1 *- form2 =
	declare2 $ \(a,b) -> row' 5 [form1 a, form2 b]

-- *| places two ref->forms in a column and joins their subject types
-- into a pair
infixr 5 *|
form1 *| form2 =
	declare2 $ \(a,b) -> column' 5 [form1 a, form2 b]


-- tab2 places two ref->forms (paired with text labels) in a notebook with
-- two tabs and joins their subject types into a pair
tab2 ::
	(String, Ref cx t1 -> FForm cx (CPanel ()) Layout) ->
	(String, Ref cx t2 -> FForm cx (CPanel ()) Layout) ->
	Ref cx (t1,t2) -> FForm cx w Layout
tab2 (title1,form1) (title2,form2) = declare2 $ \(ref1,ref2) ->
	tabs'
		[ tab' title1 $ form1 ref1
		, tab' title2 $ form2 ref2
		]

-- tab3 places three ref->forms (paired with text labels) in a notebook with
-- three tabs and joins their subject types into a triple
tab3 ::
	(String, Ref cx t1 -> FForm cx (CPanel ()) Layout) ->
	(String, Ref cx t2 -> FForm cx (CPanel ()) Layout) ->
	(String, Ref cx t3 -> FForm cx (CPanel ()) Layout) ->
	Ref cx (t1,t2,t3) -> FForm cx w Layout
tab3 (title1,form1) (title2,form2) (title3,form3) = declare3 $ \(ref1,ref2,ref3) ->
	tabs'
		[ tab' title1 $ form1 ref1
		, tab' title2 $ form2 ref2
		, tab' title3 $ form3 ref3
		]
	

-- ####### run a ref->form in a dialog ###############

run_in_dialog :: Window w -> (Ref cx cx -> FForm cx (CPanel ()) Layout) -> cx -> IO cx
run_in_dialog w to_form cx =
	do	pt <- get w position
		d <- dialog w [position := pointMove (vec 25 25) pt]
		p <- panel d []    -- introduce panel for 'default button' behaviour
		let (FForm form) = to_form idref
		(lay,upd) <- form p cx
		ok <- button p [text := "OK"]
		cancel <- button p [text := "Cancel"]
		set p [defaultButton := ok]
		let withbuttons =
			container p $ margin 10 $ column 20 [lay, alignBottomRight $ row 5 [widget ok, widget cancel]]
		set d [layout := withbuttons]
		let setstop = \stop ->
			do
				set ok [on command := do f <- upd; stop $ Just $ f cx]
				set cancel [on command := stop Nothing]
		newcx <- showModal d setstop
		return $	case newcx of
			Just ncx -> ncx
			Nothing ->  cx
